(in-package "CL-USER")

;; (load "dbmc-structs")

;like in acl2, returns nil iff x is a positive integer.
(declaim (ftype (function (t) boolean) zp))
(defun zp (x)
  (if (integerp x) (<= x 0) t))

;x is 0
(declaim (ftype (function (t) boolean) 0p))
(defun 0p (x)
  (eql x 0))

;x is 1
(declaim (ftype (function (t) boolean) 1p))
(defun 1p (x)
  (eql x 1))

;x is a positive integer
(declaim (ftype (function (t) boolean) posp))
(defun posp (x)
  (and (integerp x)
       (< 0 x)))

;x is a natural number
(declaim (ftype (function (t) boolean) natp)
	 (inline natp))
(defun natp (x)
  (and (integerp x)
       (<= 0 x)))

;x is a true list (nil-terminated list), like in acl2
(declaim (ftype (function (t) boolean) true-listp))
(defun true-listp (x)
  (if (atom x) 
      (not x)
    (true-listp (cdr x))))

;checks if x is a true-list of 1s and 0s.
(declaim (ftype (function (t) boolean) binary-listp))
(defun binary-listp (x)
  (if (atom x)
      (not x)
    (and (or (0p (first x))
             (1p (first x)))
         (binary-listp (rest x)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;---bit vector related functions---;
; bit vectors are represented as   ;
; lists of 1s and 0s here          ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;swaps whether x is 0 or 1. if x isn't either, it is considered 0.
(declaim (ftype (function (bit) bit) invb))
(defun invb (x)
  (if (equal x 0)
      1
    0))

;invert an entire bit-vector. in other words, negation.
(declaim (ftype (function (list) list) invbv))
(defun invbv (l)
  (if (endp l)
      nil
    (cons (invb (car l))
          (invbv (cdr l)))))

;ubv-n helper function
(declaim (ftype (function (list integer) integer) ubv-n-aux))
(defun ubv-n-aux (bv n)
  (if (endp bv)
      n
    (ubv-n-aux (cdr bv)
	       (+ (* 2 n) (the (member 0 1) (car bv))))))

;converts bv from an unsigned bit vector to the equivalent natural
;number
(declaim (ftype (function (list) integer) ubv-n))
(defun ubv-n (bv)
  (ubv-n-aux bv 0))

;converts a signed-bit-vector to the equivalent integer
(declaim (ftype (function (list) integer) sbv-i))
(defun sbv-i (bv)
  (if (or (endp bv)
	  (0p (car bv)))
      (ubv-n bv)
    (- (1+ (ubv-n (invbv bv))))))

;converts a natural number into the equivalent unsigned bit vector of
;minimum length.
(declaim (ftype (function (integer) list) n-ubv))
(defun n-ubv (n)
  (cond ((zp n)
         nil)
        ((evenp n)
         (append (n-ubv (/ n 2)) '(0)))
        (t (append (n-ubv (/ (1- n) 2)) '(1)))))

(declaim (ftype (function (list) (values list boolean)) negate1))
(defun negate1 (bv)
  (if (endp bv)
      (values nil nil)
    (multiple-value-bind 
	(nbv done)
	(negate1 (cdr bv))
	(cond (done
	       (cons (invb (car bv)) nbv))
	      ((1p (car bv))
	       (values (cons 1 nbv) t))
	      (t
	       (values (cons 0 nbv) nil))))))

(declaim (ftype (function (list) list) negate))
(defun negate (bv)
  (multiple-value-bind 
      (nbv done)
      (negate1 bv)
    (declare (ignore done))
    nbv))

;converts an integer to the equivalent bit vector of minimum length.
(declaim (ftype (function (integer) list) i-sbv))
(defun i-sbv (i)
  (if (natp i)
      (cons 0 (n-ubv i))
    (let ((res (negate (n-ubv (- i)))))
      (if (0p (car res))
          (cons 1 res)
        res))))


;the ceiling of the log of x base 2.
(declaim (ftype (function (integer) integer) ceil-log))
(defun ceil-log (x)
  (ceiling (log x 2)))

(declaim (ftype (function (integer) integer) floor-log))
(defun floor-log (x)
  (floor (log x 2)))

;gives the number of bits in the signed bit vector representation of i.
(declaim (ftype (function (integer) integer) size))
(defun size (i)
  (cond ((= i 0) 1)
	((< i 0) (1+ (ceil-log (abs i))))
	(t       (1+ (ceil-log (1+ i))))))
;  (length (i-sbv i)))

(declaim (ftype (function (integer) integer) usize))
(defun usize (n)
  (ceil-log (1+ n)))
;  (length (n-ubv n)))

;returns a list containing n copies of x.
(declaim (ftype (function (t fixnum) list) n-copies))
(defun n-copies (x n)
  (if (zp n)
      nil
    (cons x (n-copies x (1- n)))))

;does a signed extension of bv to size bits. that is, it copies the
;most significant bit of bv enough times to pad it.
(declaim (ftype (function (list fixnum) list) sign-extend))
(defun sign-extend (bv size)
  (let ((len (length bv))
        (sign (car bv)))
    (append (n-copies sign (- size len))
            bv)))

;unsigned extension of bv to size bits. that is, it pads bv with zeros.
(declaim (ftype (function (list fixnum) list) extend))
(defun extend (bv size)
  (append (n-copies 0 (- size (length bv)))
          bv))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;appends the item i to the end of list lst.
(declaim (ftype (function (list t) list)))
(defun app-item (lst i)
  (append lst (list i)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;gets the funct struct representing the function named funct-name from desc d.
(declaim (ftype (function (symbol desc) (or funct null)) get-funct))
(defun get-funct (funct-name d)
  (cdr (assoc funct-name (desc-functs d))))

;returns nil if and only if form isn't an integer or the form
;representation of a constant.
(declaim (ftype (function ((or integer formula)) boolean) const-formp))
(defun const-formp (form)
  (or (integerp form)
      (eq (formula-fn form) 'const)))

;returns nil iff form is not "atomic", that is, not a constant, var,
;v0-var, or next.
(declaim (ftype (function ((or integer formula)) boolean) atomic-formp))
(defun atomic-formp (form)
  (or (const-formp form)
      (eq (formula-fn form) 'var)
      (eq (formula-fn form) 'v0-var)
      (eq (formula-fn form) 'next)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; note: slot1 and slot2 are used to store results during computations    ;
; to avoid recomputation. the forms created via simplification would     ;
; be exponential in the size of the original if we didn't share          ;
; structure. likewise, most algorithms have to walk the entire           ;
; formula, but can reuse results from shared subformulas.                ;
;                                                                        ;
; any algorithm that uses either slot should set that slot in every      ;
; subform. when the algorithm is done, call the appropriate clearing     ;
; function. if you do not set the slot in every subform, some subforms   ;
; might not get cleared. this is because the clearing algorithms don't   ;
; traverse sub-formulas of formulas whose slot is already set to         ;
; nil. if they didn't do this, they would have exponential running time. ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;clears the slot1 field of form and all its subforms. also returns the
;cleaned form.
(declaim (ftype (function (t) null) clear-slot1-aux))
(defun clear-slot1-aux (form)
  (cond ((vec-p form)
	 (dotimes (i (vec-num-bits form))
	   (clear-slot1-aux (vec-get-bit form i))))
	((formula-p form)
	 (when (formula-slot1 form) 
	   (setf (formula-slot1 form) nil)
	   (mapcar #'clear-slot1-aux
		   (formula-args form)))
	 nil)
	((mem-p form)
	 (dotimes (i (mem-num-words form))
	   (clear-slot1-aux (mem-get-word form i))))))

(declaim (ftype (or (function (formula) formula)
		    (function (vec) vec)
		    (function (mem) mem))
		clear-slot1)
	 (inline clear-slot1))
(defun clear-slot1 (form)
  (clear-slot1-aux form)
  form)

;clears the slot2 field of form and all its subforms. also returns the
;cleaned form.
(declaim (ftype (function (t) null) clear-slot2-aux))
(defun clear-slot2-aux (form)
  (cond ((vec-p form)
	 (dotimes (i (vec-num-bits form)) (clear-slot2-aux (vec-get-bit form i))))
	((formula-p form)
	 (when (formula-slot2 form) 
	   (setf (formula-slot2 form) nil)
	   (mapcar #'clear-slot2-aux
		   (formula-args form)))
	 nil)
	((mem-p form)
	 (dotimes (i (mem-num-words form))
	   (clear-slot2-aux (mem-get-word form i))))))

(declaim (ftype (or (function (formula) formula)
		    (function (vec) vec)
		    (function (mem) mem))
		clear-slot2)
	 (inline clear-slot2))
(defun clear-slot2 (form)
  (clear-slot2-aux form)
  form)

;clears both slot fields of form and all its subforms. also returns the
;cleaned form.
(declaim (ftype (function (t) null) clear-both-slots-aux))
(defun clear-both-slots-aux (form)
  (cond ((vec-p form)
	 (dotimes (i (vec-num-bits form)) (clear-both-slots-aux (vec-get-bit form i))))
	((formula-p form)
	 (when (or (formula-slot1 form)
		   (formula-slot2 form))
	   (setf (formula-slot1 form) nil)
	   (setf (formula-slot2 form) nil)
	   (mapcar #'clear-both-slots-aux
		   (formula-args form)))
	 nil)
	((mem-p form)
	 (dotimes (i (mem-num-words form))
	   (clear-both-slots-aux (mem-get-word form i))))))

(declaim (ftype (or (function (formula) formula)
		    (function (vec) vec)
		    (function (mem) mem))
		clear-both-slots)
	 (inline clear-both-slots))
(defun clear-both-slots (form)
  (clear-both-slots-aux form)
  form)

(declaim (ftype (function (t) null) scrub-slots-aux))
(defun scrub-slots-aux (form)
  (cond ((vec-p form)
	 (dotimes (i (vec-num-bits form))
	   (scrub-slots-aux (vec-get-bit form i))))
	((mem-p form)
	 (dotimes (i (mem-num-words form))
	   (dotimes (j (mem-wordsize form))
	     (scrub-slots-aux (mem-get-bit form i j)))))
	((not (formula-p form)) nil)
	((eq (formula-slot1 form) 'scrub) nil)
	(t
	 (setf (formula-slot1 form) 'scrub)
	 (setf (formula-slot2 form) nil)
	 (mapcar #'scrub-slots-aux
		 (formula-args form))
	 nil)))

(declaim (ftype (or (function (formula) formula)
		    (function (vec) vec)
		    (function (mem) mem))
		scrub-slots)
	 (inline scrub-slots))
(defun scrub-slots (form)
  (scrub-slots-aux form)
  (clear-slot1 form))

(declaim (ftype (function (list) list) sortforms))
(defun sortforms (forms)
  (declare (type list forms))
  (sort (copy-list forms) #'> :key (lambda (x)
				     (declare (type form-vec x))
				     (if (vec-p x) (vec-value x) (formula-value x)))))

(declaim (ftype (function (form-type) integer) type-bits))
(defun type-bits (tp)
  (declare (type form-type tp))
  (case (car tp)
    (bv (the integer (second tp)))
    (mem (* (the integer (second tp)) (the integer (third tp))))
    (otherwise
     (format t "~&type-bits: weird type: ~a~%" tp)
     0)))

(declaim (ftype (function (formula) integer) formula-bits))
(defun formula-bits (form)
  (declare (type formula form))
  (type-bits (the form-type (formula-type form))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declaim (ftype (function (formula) null) find-slot2-residue1))
(defun find-slot2-residue1 (form)
  (cond ((not (formula-p form)) nil)
	((eq (formula-slot2 form) 'fs2r) nil)
	((formula-slot2 form) (break (format nil "slot2 residue: ~A" (formula-slot2 form))))
	(t
	 (setf (formula-slot2 form) 'fs2r)
	 (mapcar #'find-slot2-residue1 (formula-args form)))))

(declaim (ftype (function (formula) formula) find-slot2-residue))
(defun find-slot2-residue (form)
  (declare (type formula form))
  (find-slot2-residue1 form)
  (the formula (clear-slot2 (the formula form))))
